// ***********************************************************************************
// freq_counter.pde
// 
//      2011 ZeroFossilFuel and Zero Labs
// 
//     This program is free software: you can redistribute it and/or modify
//     it under the terms of the GNU General Public License as published by
//     the Free Software Foundation, either version 3 of the License, or
//     (at your option) any later version.
// 
//     This program is distributed in the hope that it will be useful,
//     but WITHOUT ANY WARRANTY; without even the implied warranty of
//     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//     GNU General Public License for more details.
// 
//     You should have received a copy of the GNU General Public License
//     along with this program.  If not, see <http://www.gnu.org/licenses/>.
//
//     Your use of this program signifies your consent to the terms above
//     license in its' entirety, both as stated here and online.
// 
// freq_counter.pde
// 
// The purpose of this program is to produce a stream of pulses whose pulse period
// is a percentage of the pulse period of an incoming pulse stream.
// 
// The percentage of the pulse period taken from the incomping pulse stream is user 
// selectable and established at compile time prior to downloading to the Arduino.
//
// Example: The table below shows pulse characteristics for an incoming pulse stream
//          along with the coresponding characteristics of the outgoing pulse stream
//          produced by this program. All durations are in microseconds.
//
//                 INCOMING PULSES                            OUTGOING PULSES
//          ==========================================================================
//          Frequency    Period    Duty %    Dwell %     Frequency    Period    Duty %
//             100 Hz      0.01      50        10           200 Hz     0.001     50
//             100 Hz      0.01      50        50           200 Hz     0.005     50
//             100 Hz      0.01      50        90           200 Hz     0.009     50
//             500 Hz     0.002      50        10          1000 Hz    0.0002     50
//             500 Hz     0.002      50        50          1000 Hz     0.001     50
//             500 Hz     0.002      50        90          1000 Hz    0.0018     50
//          ==========================================================================
//
// Implementation:
//          This program monitors the duration in microseconds of the HIGH and LOW 
//          pulse levels of an incoming stream of pulses on digital pin 2. The method
//          used to monitor these levels is through the use of 2 interrupt service 
//          routines (ISR) set to activate on the RISING and FALLING edges of each
//          pulse in the input stream. Each ISR hands control to the other at the end
//          of it's processing. Each ISR maintains a running measurement for it's
//          respective level - HIGH or LOW. This goes on indefinately with each ISR
//          passing control back to the other in a ping-pong fashion, adapting to the
//          variable frequency of the incoming pulse stream.
//
//          The remainder of the program continually adds the HIGH and LOW levels
//          whose sum is the pulse period of the incoming pulse stream. This pulse
//          period is then used to calculate a 'dwell' period based on a percentage
//          supplied by the user.
//
// Assumptions:
//          The 'dwell' period is a user supplied percentage of the incoming pulse 
//          period with a 50% duty cycle
//
// Summary:
//          Input pulse amplitude 0-5V.
//          Output pulse amplitude 0-5V.
//          Connect input pulse stream 1 to pin 2.
//          Connect input pulse stream 2 to pin 3.
//          Output pulse stream 1 from pin 7.
//          Output pulse stream 2 from pin 8.
//          Leave 16MHz system clock alone.
//          Developed and tested on AVR Studio 4.18.
//          Tested on Arduino Duemilanove & Arduino IDE 0022.
//          Variable names used by an ISR are preceded with '__' (double underscore)
// 
//          Note: Input frequency and output accuracy are inversley proportional. 
//                Therefore, the operating parameters need to be chosen such that the 
//                accuracy of the results fall within the project requirements.
// 
// 
//               50760 |*     
//                     | *     
//          Input      |    *
//          Frequency  |        *
//          Hz         |              *
//                     |                     *
//                     |                               *
//                     |                                              *
//                     |                                                                        *
//                   9 |_ _ _ _ _ _ _ _ _ _ _ _ _ _ _ _ _ _ _ __ _ _ _  _ _ _ _ _ _ _ _ _ _ _ _ _
//                     98%                                                                    100%
//                                             Dwell Percentage Accuracy
// 
// Last updated:
//          05/08/2011 - Hydrtec, initial release.
//                       Tested dwell 0.02% @ 540 RPM (9Hz) (hunting +- 4 us).
//                       Tested dwell 2% @ ? 56760 RPM (946Hz) (hunting +- 4 us).
//                       Started version number system for first milestone (v1.0)
//
//          06/08/2011 - v1.0a
//                       Added facility not to assume 50% duty cycle on the input pulses.
//                       Implemented output pulse triggering from interrupt 
//                       Implemented 'pulse forming' for full delay range support
//
//          13/08/2011 - v1.0b
//                       Removed the floating point from the interrupt calculations and 
//                       replaced them with bit-shifts and one compile-time pre-calculation.
//                       This is the start of the global optimization phase.
//
//          17/08/2011 - v1.2c
//                       Added additional code to handle integer based calculations when
//                       potentiometers are used.
//                       Added keyboard input as follows:
//                     
//                           q  =  delay1 + 0.01%
//                           Q  =  dwell1 + 0.01%
//                           a  =  delay1 - 0.01%
//                           A  =  dwell1 - 0.01%
//                           w  =  delay2 + 0.01%
//                           W  =  dwell2 + 0.01%
//                           s  =  delay2 - 0.01%
//                           S  =  dwell2 - 0.01%
//                           p  =  print settings report to serial port
//                           P  =  p
//                           r  =  Reste Arduino
//                           R  =  r
//
// Author:  Hydrtec
// Email:   hydrtec@hydrtec.com
// ***********************************************************************************

// ***********************************************************************************
// User controls.
// ***********************************************************************************
volatile float dwell_percentage1 = 50;       // 99% MAXIMUM for now... (1)
volatile float dwell_percentage2 = 50;       // 99% MAXIMUM for now... (2)
volatile float dwell_delay_percentage1 = 0;  //  99% MAXIMUM for now...(1)
volatile float dwell_delay_percentage2 = 0;  //  99% MAXIMUM for now...(2)
// ***********************************************************************************
// End of user controls.
// ***********************************************************************************

//#define DEBUG                            // Enable debug ouput - at cost/accuracy
//#define DEBUG_ANALOG_ONLY                // Enable debug ouput - analog only
//#define VARIABLE_HIGH                    // Dwell HIGH is percentage of input period

#define READ_ANALOG_DELAY1               // Allow setting of delay from Potentiometers
#define READ_ANALOG_DELAY2               // instead of the delay constants shown above
#define READ_ANALOG_DWELL1               // Allow setting of dwell from Potentiometers
#define READ_ANALOG_DWELL2               // instead of the dwell constants shown above

#define MAX_MISSED_PULSES 100              // Maximum number of miseed pulses before an 
                                           // error condition is flagged

//#define RESET_ON_MISSED_PULSES             // If missed pulses recur, then reboot

#define USE_KEYBOARD                       // Enable keyboard input
#define USE_KEYBOARD_STATS                 // Enable keyboard output to serial port

// Leave this here for the time being until I decide what to do!
#if defined(USE_KEYBOARD) || defined(USE_KEYBOARD_STATS)
#ifdef READ_ANALOG_DELAY1
#undef READ_ANALOG_DELAY1
#endif
#ifdef READ_ANALOG_DELAY2
#undef READ_ANALOG_DELAY2
#endif
#ifdef READ_ANALOG_DWELL1
#undef READ_ANALOG_DWELL1
#endif
#ifdef READ_ANALOG_DWELL2
#undef READ_ANALOG_DWELL2
#endif
#endif

// ***********************************************************************************
// If using AVR Studio, remember to un-comment the line #define AVR_STUDIO, otherwise
// the Arduino IDE is assumed as the development environment.
// 
// Command line parameters to upload to Arduino Duemilanove assuming 57600 baud rate:
//
// avrdude -p m328p -c avrisp -P com3 -b 57600 -F -U flash:w:freq_counter.hex
// 
// Press RESET on the Arduino within 2 seconds of pressing ENTER!!!
//
// Note: The '.hex' file is located in a folder called 'default' which is created 
// inside the project folder by AVR Studio. Substitute 'com3' with your own com port.
// ***********************************************************************************
//#define AVR_STUDIO
#ifdef AVR_STUDIO
#define F_CPU 16000000UL
#endif

#include "WProgram.h"
#include <avr/wdt.h>

#ifdef AVR_STUDIO
#if defined(DEBUG) || defined(DEBUG_ANALOG_ONLY) || defined(USE_KEYBOARD_STATS)
#include "./arduino/libraries/NewSoftSerial/NewSoftSerial.h"
NewSoftSerial mySerial(0, 1);
#define IOSERIAL mySerial
#endif
#else
#if defined(DEBUG) || defined(DEBUG_ANALOG_ONLY) || defined(USE_KEYBOARD_STATS)
#define IOSERIAL Serial
#endif
#endif

#define FMULT  1024
#define FSHIFT 10

// ***********************************************************************************
// Experimental...pulse level inversion section
// ***********************************************************************************
//#define INVERT_OUTPUT1                     // Invert output HIGH/LOW
//#define INVERT_OUTPUT2                     // Invert output HIGH/LOW

// ***********************************************************************************
// Forward declarations to keep AVR Studio happy. Arduino 'language' is less strict
// and as a consequence leads to some pretty bad habits to be developed by beginners.
// ***********************************************************************************
void setup(void);
void ISR_CHANGE_0(void);
void ISR_CHANGE_1(void);
boolean isPulseReady(int pin);
unsigned long getTicksHigh1(void);
unsigned long getTicksLow1(void);
unsigned long getTicksHigh2(void);
unsigned long getTicksLow2(void);
unsigned long getPulseIn(int pin, boolean level);
void outputPulse1(void);
void outputPulse2(void);
void errorCheck(void);
void loop(void);

// ***********************************************************************************
// Experimental...pulse level inversion section
// ***********************************************************************************

#ifdef INVERT_OUTPUT1
#define OUT_HIGH1 LOW
#define OUT_LOW1  HIGH
#else
#define OUT_HIGH1 HIGH
#define OUT_LOW1  LOW
#endif

#ifdef INVERT_OUTPUT2
#define OUT_HIGH2 LOW
#define OUT_LOW2  HIGH
#else
#define OUT_HIGH2 HIGH
#define OUT_LOW2  LOW
#endif

// ***********************************************************************************
// These constants defines values expressed in microseconds. They are used to make
// small calibration adjustments of measured microseconds during the pulse capture 
// phases in order to match the oscilloscope readings I get. Of course, my scope 
// ***might*** be out of calibration but it is a Pico 200/100 and seems to be very 
// accurate in everything else when I compare it with the readings from my Tektronix 475 
// and that is ***not*** out of calibration. Nevertherless...experimental only!!!
// ***********************************************************************************
#define FREQ_ADJUST1 0
#define FREQ_ADJUST2 0

// ***********************************************************************************
// Minimal difference equates to get AVR Studio and Arduino 'IDE' to share code.
// ***********************************************************************************
#ifdef AVR_STUDIO
#define A0 14
#define A1 15
#define A2 16
#define A3 17
#define A4 18
#define A5 19
#define A6 20
#define A7 21
#endif

// ***********************************************************************************
// Pin definitions section
// ***********************************************************************************
const int inputPin1 = 2;
const int inputPin2 = 3;
#ifdef READ_ANALOG_DELAY1
const int inputAPin1 = A0;
#endif
#ifdef READ_ANALOG_DELAY2
const int inputAPin2 = A1;
#endif
#ifdef READ_ANALOG_DWELL1
const int inputAPin3 = A2;
#endif
#ifdef READ_ANALOG_DWELL2
const int inputAPin4 = A3;
#endif
const int outputPin1 = 7;
const int outputPin2 = 8;

// ***********************************************************************************
// PS2Keyboard datapin and IRQ pin definiations
// ***********************************************************************************
const int KBD_DataPin = 4;
const int KBD_IRQpin  = 6;

// ***********************************************************************************
// Flags used for control and syncronization. This is the only 'running' coupling.
// ***********************************************************************************
volatile boolean Pout1 = false;    // Process control toggle (1)
volatile boolean Pout2 = false;    // Process control toggle (2) 

// ***********************************************************************************
// These variables hold the measured time in microseconds for the HIGH and LOW level
// of the incoming pulses on pin 1. They are constantly updated in interrupt handler.
// ***********************************************************************************
volatile unsigned long __timer_ticks_high0 = 0;        // HIGH level microseconds (1)
volatile unsigned long __timer_ticks_low0 = 0;         // LOW level microseconds (1) 
volatile boolean __got_pulse0 = false;                 // Pulse aquisition flag (1)
volatile unsigned long __pulse_start_time0 = 0;        // Pulse start microsecond (1)
volatile unsigned long __pulse_completion_time0 = 0;   // Pulse completion microsecond (1)
volatile unsigned long __iperiod0 = 0;                 // Pulse period microseconds (1)
volatile unsigned long __pulse_mark0 = 0;              // Time marking (input 1)
volatile int __pulse_event0 = 0;
volatile int __pulse_count0 = 0;

// ***********************************************************************************
// These variables hold the measured time in microseconds for the HIGH and LOW level
// of the incoming pulses on pin 2. They are constantly updated in interrupt handler.
// ***********************************************************************************
volatile unsigned long __timer_ticks_high1 = 0;        // HIGH level microseconds (2)
volatile unsigned long __timer_ticks_low1 = 0;         // LOW level microseconds (2) 
volatile boolean __got_pulse1 = false;                 // Pulse aquisition flag (2)
volatile unsigned long __pulse_start_time1 = 0;        // Pulse start microsecond (2)
volatile unsigned long __pulse_completion_time1 = 0;   // Pulse completion microsecond (2)
volatile unsigned long __iperiod1 = 0;                 // Pulse period microseconds (2)
volatile unsigned long __pulse_mark1 = 0;              // Time marking (input 2)
volatile int __pulse_event1 = 0;
volatile int __pulse_count1 = 0;

// ***********************************************************************************
// These variables are used to hold the HIGH and LOW levels of the incoming pulses
// on input pin 1.
// ***********************************************************************************
volatile unsigned long pulseWidthHigh1 = 0;            // Intermediate pulse HIGH (1)
volatile unsigned long pulseWidthLow1 = 0;             // Intermediate pulse LOW (1)
volatile unsigned long pulsePeriod1 = 0;               // Intermediate pulse period (1)

// ***********************************************************************************
// These variables are used to hold the HIGH and LOW levels of the incoming pulses
// on input pin 2.
// ***********************************************************************************
volatile unsigned long pulseWidthHigh2 = 0;            // Intermediate pulse HIGH (2)
volatile unsigned long pulseWidthLow2 = 0;             // Intermediate pulse LOW (2)
volatile unsigned long pulsePeriod2 = 0;               // Intermediate pulse period (2)


// ***********************************************************************************
// Intermediate output pulse level durations section 
// ***********************************************************************************
unsigned long dwell_high1 = 0;                         // Intermediate HIGH (input 1)
unsigned long dwell_low1 = 0;                          // Intermediate LOW (input 1)
unsigned long dwell_delay1 = 0;                        // Intermediate delay (input 1)
unsigned long dwell_high2 = 0;                         // Intermediate HIGH (input 2)
unsigned long dwell_low2 = 0;                          // Intermediate LOW (input 2)
unsigned long dwell_delay2 = 0;                        // Intermediate delay (input 2)

// ***********************************************************************************
// Output levels flag section
// ***********************************************************************************
boolean wait_preambleHigh1 = true;                     // Delay preamble flag (output 1)
boolean wait_delay1 = true;                            // Delay procesing flag (output 1)
boolean wait_high1 = true;                             // HIGH processing flag (output 1)
boolean wait_low1 = true;                              // LOW processing flag (output 1)
boolean wait_preambleHigh2 = true;                     // Delay preamble flag (output 2)
boolean wait_delay2 = true;                            // Delay processing flag (output 2)
boolean wait_high2 = true;                             // HIGH processing flag (output 2)
boolean wait_low2 = true;                              // LOW processing flag (output 2)

volatile unsigned long preambleHigh1 = 0;              // Preamble logic level (1)
volatile unsigned long preambleHigh2 = 0;              // Preamble logic level (2)

// ***********************************************************************************
// Global pulse time marker
// ***********************************************************************************
unsigned long time_mark = 0;                           // Global time marker

// ***********************************************************************************
// Local ISR pulse time markers
// ***********************************************************************************
int pulse_count1 = 1;
int pulse_count2 = 1;

// ***********************************************************************************
// Analog input section
// ***********************************************************************************
#ifdef READ_ANALOG_DELAY1
volatile float analog_delay1 = 0;  // Analog delay voltage reading (1)
volatile float old_analog_delay1 = 1;  // Analog delay voltage reading (1)
#endif
#ifdef READ_ANALOG_DELAY2
volatile float analog_delay2 = 0;  // Analog delay voltage reading (2)
volatile float old_analog_delay2 = 1;  // Analog delay voltage reading (2)
#endif

#ifdef READ_ANALOG_DWELL1
volatile float analog_dwell1 = 0;  // Analog dwell voltage reading (1)
volatile float old_analog_dwell1 = 1;  // Analog dwell voltage reading (1)
#endif
#ifdef READ_ANALOG_DWELL2
volatile float analog_dwell2 = 0;  // Analog dwell voltage reading (2)
volatile float old_analog_dwell2 = 1;  // Analog dwell voltage reading (2)
#endif

#if defined(READ_ANALOG_DELAY1) || defined(READ_ANALOG_DELAY1) || defined(READ_ANALOG_DELAY2) || defined(READ_ANALOG_DELAY2)
void processAnalogInputs(void);
#endif

volatile unsigned long uldwell_percentage1 = 50;       // ... (1)
volatile unsigned long uldwell_percentage2 = 50;       // ... (2)
volatile unsigned long uldwell_delay_percentage1 = 0;  // ... (1)
volatile unsigned long uldwell_delay_percentage2 = 0;  // ... (2)

// ***********************************************************************************
// Reset method 1 using the jump vector. This restarts the program ***but*** register 
// status remains unchanged across the reset.
// ***********************************************************************************
void(* softReset1) (void) = 0; //declare reset function @ address 0

// ***********************************************************************************
// PS2Keyboard header first...
// ***********************************************************************************
#ifdef  USE_KEYBOARD
#include <PS2Keyboard.h>

// ***********************************************************************************
// Instantiate an instance of the PS2Kboard class
// ***********************************************************************************
PS2Keyboard keyboard;
#endif

// ***********************************************************************************
// void setup(void)
// 
// Purpose:   Initialise variable and porogram running parameters
// 
// Inputs:    none
// 
// Returns:   nothing
// ***********************************************************************************
void setup(void)
{
  // ************************************************
  // Fire up the keyboard after a one second delay
  // ************************************************
  delay(1000);
#if defined(DEBUG) || defined(DEBUG_ANALOG_ONLY) || defined(DEBUG2) || defined(USE_KEYBOARD_STATS)
  IOSERIAL.begin(57600);
#endif

#ifdef  USE_KEYBOARD
  keyboard.begin(KBD_DataPin, KBD_IRQpin);
#endif

  pinMode(inputPin1, INPUT);        // Input pulses 1
  pinMode(inputPin2, INPUT);        // Input pulses 2
  digitalWrite(inputPin1, HIGH);    // Enable internal pullup for this pin (1)
  digitalWrite(inputPin2, HIGH);    // Enable internal pullup for this pin (2)
  pinMode(outputPin1, OUTPUT);
  pinMode(outputPin2, OUTPUT);

  // ************************************************
  // Make sure the coupling flags are cleared.
  // ************************************************
  Pout1 = true;
  Pout2 = true;
  
  // ************************************************
  // Pre-calculate dwell % to remove from loop
  // ************************************************
  dwell_percentage1 = (dwell_percentage1 / 100.0);              // Pre-calculate dwell percentage division (1)
  dwell_percentage2 = (dwell_percentage2 / 100.0);              // Pre-calculate dwell percentage division (2) 
  dwell_delay_percentage1 = (dwell_delay_percentage1 / 100.0);  // Pre-calculate dwell delay percentage division (1)
  dwell_delay_percentage2 = (dwell_delay_percentage2 / 100.0);  // Pre-calculate dwell delay percentage division (2)

  uldwell_percentage1 = (unsigned long)(dwell_percentage1 * FMULT);              // Pre-calculate dwell percentage division (1)
  uldwell_percentage2 = (unsigned long)(dwell_percentage2 * FMULT);              // Pre-calculate dwell percentage division (2) 
  uldwell_delay_percentage1 = (unsigned long)(dwell_delay_percentage1 * FMULT);  // Pre-calculate dwell delay percentage division (1)
  uldwell_delay_percentage2 = (unsigned long)(dwell_delay_percentage2 * FMULT);  // Pre-calculate dwell delay percentage division (2)

  // ************************************************
  // Initialise variables for stream 0
  // ************************************************
  __timer_ticks_low0 = 0;          // Pulse LOW duration (input 1)
  __timer_ticks_high0 = 0;         // Pulse HIGH duration (input 1)
  __pulse_start_time0 = 0;         // Start of pulse in microseconds (input 1)
  __pulse_completion_time0 = 0;    // End of pulse in microseconds (input 1a)
  __iperiod0 = 0;                  // Pulse period (input 1)
  __pulse_event0 = 0;
  __pulse_count0 = 0;
  
  // ************************************************
  // Initialise variables for stream 1
  // ************************************************
  __timer_ticks_low1 = 0;          // Pulse LOW duration (input 2)
  __timer_ticks_high1 = 0;         // Pulse HIGH duration (input 2)
  __pulse_start_time1 = 0;         // Start of pulse in microseconds (input 2)
  __pulse_completion_time1 = 0;    // End of pulse in microseconds (input 2a)
  __iperiod1 = 0;                  // Pulse period (input 2)
  __pulse_event1 = 0;
  __pulse_count1 = 0;
  
  // ************************************************
  // Working variables
  // ************************************************
  dwell_high1 = 0;                 // Intermediate HIGH (input 1)
  dwell_low1 = 0;                  // Intermediate LOW (input 1)
  dwell_delay1 = 0;                // Intermediate delay (input 1)
  dwell_high2 = 0;                 // Intermediate HIGH (input 2)
  dwell_low2 = 0;                  // Intermediate LOW (input 2)
  dwell_delay2 = 0;                // Intermediate delay (input 2)
  
  wait_preambleHigh1 = true;       // Delay procesing flag (output 1)
  wait_delay1 = true;              // Delay procesing flag (output 1)
  wait_high1 = true;               // HIGH processing flag (output 1)
  wait_low1 = true;                // LOW processing flag (output 1)
  wait_preambleHigh2 = true;       // Delay procesing flag (output 1)
  wait_delay2 = true;              // Delay processing flag (output 2)
  wait_high2 = true;               // HIGH processing flag (output 2)
  wait_low2 = true;                // LOW processing flag (output 2)

  __pulse_mark0 = 0;               // Time marking (input 1)
  __pulse_mark1 = 0;               // Time marker (input 2)

  // ************************************************
  // Initialize global pulse time marker
  // ************************************************
  time_mark = 0;                   // Global time marker

  // ************************************************
  // Initialize local ISR pulse time markers
  // ************************************************
  pulse_count1 = 1;                // Pulse count (1)
  pulse_count2 = 1;                // Pulse count (2)
  
  // ************************************************
  // Analog variable section. Hold potentiometer
  // readings prior to convertion to percentage(s).
  // ************************************************
#ifdef READ_ANALOG_DELAY1
  analog_delay1 = 0;               // Analog delay voltage reading (output 1)
  old_analog_delay1 = 1;               // Analog delay voltage reading (output 1)
#endif
#ifdef READ_ANALOG_DELAY2
  analog_delay2 = 0;               // Analog delay voltage reading (output 2)
  old_analog_delay2 = 1;               // Analog delay voltage reading (output 2)
#endif

#ifdef READ_ANALOG_DWELL1
  analog_dwell1 = 0;               // Analog dwell voltage reading (output 1)
  old_analog_dwell1 = 1;               // Analog dwell voltage reading (output 1)
#endif
#ifdef READ_ANALOG_DWELL2
  analog_dwell2 = 0;               // Analog dwell voltage reading (output 2)
  old_analog_dwell2 = 1;               // Analog dwell voltage reading (output 2)
#endif

  // ************************************************
  // Todo: 
  // ************************************************
  attachInterrupt(0, ISR_CHANGE_0, CHANGE);  // Start first ISR (input 1)
  attachInterrupt(1, ISR_CHANGE_1, CHANGE);  // Start second ISR (input 2)

#if defined(DEBUG) || defined(DEBUG_ANALOG_ONLY)
  int x = 25;
  while(x--) {
    IOSERIAL.println();
  }
  IOSERIAL.println("Muller Motor Driver");
  IOSERIAL.println("Version 1.0");
  IOSERIAL.println();
#endif
  
  // *************************************************************
  // Enable and give the pulse capture functions time to lock.
  // *************************************************************
  for(int x=0;x<10;x++) {
    Pout1 = false;
    Pout2 = false;
    while(!isPulseReady(0) && !isPulseReady(1));
  }
}

// ***********************************************************************************
// void ISR_CHANGE_0(void)
// 
// Purpose:   Maintains a running duration of the incoming pulse periods. This is an
//            interrupt service routine and is triggered by the LEVEL CHANGE of the 
//            incoming pulse on inputPin2.
// 
//            Also, 'form' the outut pulse depending on the sum of the delay and dwell
//            durations exceeding the input pulse period. The intersection (difference)
//            is used to create a 'pre-delay' logic level and then trim the dwell & delay.
// 
// Inputs:    None
// 
// Returns:   Nothing
// ***********************************************************************************
void ISR_CHANGE_0(void)
{
  if(digitalRead(inputPin1) != HIGH) {
    return;
  }
  
  wait_delay1 = false;
  wait_high1 = false;
  wait_low1 = false;
    
  if(__pulse_count0 == 0) {
    __pulse_start_time0 = micros();
    __pulse_completion_time0 = 0;
    __got_pulse0 = false;
    __pulse_count0++;
    return;
  }
  else if(__pulse_count0 == 1) {
    __pulse_completion_time0 = micros();
    __got_pulse0 = false;
    __pulse_count0++;
    return;
  }

  __pulse_count0++;

  __iperiod0 = (__pulse_completion_time0 - __pulse_start_time0);
  __pulse_start_time0 = __pulse_completion_time0;
  __pulse_completion_time0 = micros();
  
  __pulse_mark0 = __pulse_completion_time0;
  __got_pulse0 = true;

  // *************************************************************
  // 1. Calculate the percentage 'dwell' and dealy.
  // *************************************************************
  dwell_high1 = ((__iperiod0 * uldwell_percentage1) >> FSHIFT) - FREQ_ADJUST1;
  dwell_delay1 = ((__iperiod0 * uldwell_delay_percentage1) >> FSHIFT) - FREQ_ADJUST1;
  // *************************************************************
  // 2. Calculate the HIGH duration of the 'dwell' period. 
  // *************************************************************
  __timer_ticks_high0 = dwell_high1;
  // *************************************************************
  // 3. Calculate the LOW duration of the 'dwell' period. 
  // *************************************************************
#ifdef VARIABLE_HIGH
  __timer_ticks_low0 = __iperiod0 - dwell_high1;
#else
  __timer_ticks_low0 = __timer_ticks_high0;
#endif

  preambleHigh1 = (dwell_delay1 + dwell_high1);
  if(preambleHigh1 >= __iperiod0) {
    preambleHigh1 = preambleHigh1 - __iperiod0;
    dwell_high1 = dwell_high1 - preambleHigh1;
    dwell_delay1 = dwell_delay1 - preambleHigh1;
    digitalWrite(outputPin1, OUT_HIGH1);
    wait_preambleHigh1 = true;
  } else {
    wait_preambleHigh1 = false;
  }
  
  wait_delay1 = true;
  wait_high1 = true;
  wait_low1 = true;
}

// ***********************************************************************************
// void ISR_CHANGE_1(void)
// 
// Purpose:   Maintains a running duration of the incoming pulse periods. This is an
//            interrupt service routine and is triggered by the LEVEL CHANGE of the 
//            incoming pulse on inputPin1.
// 
//            Also, 'form' the outut pulse depending on the sum of the delay and dwell
//            durations exceeding the input pulse period. The intersection (difference)
//            is used to create a 'pre-delay' logic level and then trim the dwell & delay.
// 
// Inputs:    None
// 
// Returns:   Nothing
// ***********************************************************************************
void ISR_CHANGE_1(void)
{
  if(digitalRead(inputPin2) != HIGH) {
    return;
  }
  
  if(__pulse_count1 == 0) {
    __pulse_start_time1 = micros();
    __pulse_completion_time1 = 0;
    __got_pulse1 = false;
    __pulse_count1++;
    return;
  }
  else if(__pulse_count1 == 1) {
    __pulse_completion_time1 = micros();
    __got_pulse1 = false;
    __pulse_count1++;
    return;
  }
  
  __pulse_count1++;
  
  wait_delay2 = false;
  wait_high2 = false;
  wait_low2 = false;
    
  __iperiod1 = (__pulse_completion_time1 - __pulse_start_time1);
  __pulse_start_time1 = __pulse_completion_time1;
  __pulse_completion_time1 = micros();
  
  __pulse_mark1 = __pulse_completion_time1;
  __got_pulse1 = true;

  // *************************************************************
  // 1. Calculate the percentage 'dwell' and dealy.
  // *************************************************************
  dwell_high2 = ((__iperiod1 * uldwell_percentage2) >> FSHIFT) - FREQ_ADJUST2;
  dwell_delay2 = ((__iperiod1 * uldwell_delay_percentage2) >> FSHIFT) - FREQ_ADJUST2;
  // *************************************************************
  // 2. Calculate the HIGH duration of the 'dwell' period. 
  // *************************************************************
  __timer_ticks_high1 = dwell_high2;
  // *************************************************************
  // 3. Calculate the LOW duration of the 'dwell' period. 
  // *************************************************************
#ifdef VARIABLE_HIGH
  __timer_ticks_low1 = __iperiod1 - dwell_high2;
#else
  __timer_ticks_low1 = __timer_ticks_high1;
#endif

  preambleHigh2 = (dwell_delay2 + dwell_high2);
  if(preambleHigh2 >= __iperiod1) {
    preambleHigh2 = preambleHigh2 - __iperiod1;
    dwell_high2 = dwell_high2 - preambleHigh2;
    dwell_delay2 = dwell_delay2 - preambleHigh2;
    digitalWrite(outputPin2, OUT_HIGH2);
    wait_preambleHigh2 = true;
  } else {
    wait_preambleHigh2 = false;
  }
  
  wait_delay2 = true;
  wait_high2 = true;
  wait_low2 = true;
}

// ***********************************************************************************
// boolean isPulseReady(int pin)
// 
// Purpose:   Reports if a pulse detection is complete for a given pin.
// 
// Inputs:    pin - integer 0 or 1
// 
// Returns:   TRUE or FALSE
// ***********************************************************************************
boolean isPulseReady(int pin)
{
  switch(pin) {
    case 0:
      return(__got_pulse0);
      break;
    case 1:
      return(__got_pulse1);
      break;
    default:
      break;
  }
  return(false);
}

// ***********************************************************************************
// unsigned long getTicksHigh1(void)
// 
// Purpose:   Return the last calculated duration in microseconds of the HIGH level of
//            a pulse period.
//
// Inputs:    None
// 
// Returns:   Stored duration in microseconds of HIGH pulse level
// ***********************************************************************************
unsigned long getTicksHigh1(void)
{
  return(__timer_ticks_high0);
}

// ***********************************************************************************
// unsigned long getTicksLow1(void)
// 
// Purpose:   Return the last calculated duration in microseconds of the LOW level of
//            a pulse period.
// 
// Inputs:    None
// 
// Returns:   Stored duration in microseconds of LOW pulse level
// ***********************************************************************************
unsigned long getTicksLow1(void)
{
  return(__timer_ticks_low0);
}

// ***********************************************************************************
// unsigned long getTicksHigh2(void)
// 
// Purpose:   Return the last calculated duration in microseconds of the HIGH level of
//            a pulse period.
//
// Inputs:    None
// 
// Returns:   Stored duration in microseconds of HIGH pulse level
// ***********************************************************************************
unsigned long getTicksHigh2(void)
{
  return(__timer_ticks_high1);
}

// ***********************************************************************************
// unsigned long getTicksLow2(void)
// 
// Purpose:   Return the last calculated duration in microseconds of the LOW level of
//            a pulse period.
// 
// Inputs:    None
// 
// Returns:   Stored duration in microseconds of LOW pulse level
// ***********************************************************************************
unsigned long getTicksLow2(void)
{
  return(__timer_ticks_low1);
}

// ***********************************************************************************
// unsigned long getPulseIn(int pin, boolean level)
// 
// Purpose:   Return the HIGH or LOW level duration in microseconds of the last pulse
//            period based on the input parameters.
// 
// Inputs:    pin - 0 = pin 2, 1 = pin 3
//            level - 0 = HIGH, 1 = LOW
// 
// Returns:   HIGH or LOW duration of pulse period
// ***********************************************************************************
unsigned long getPulseIn(int pin, boolean level) 
{
  switch(pin) {
    case 0:
      switch(level) {
        case HIGH:
          return(__timer_ticks_high0);
          break;
        case LOW:
          return(__timer_ticks_low0);
          break;
        default:
          break;
      }
      break;
    case 1:
      switch(level) {
        case HIGH:
          return(__timer_ticks_high1);
          break;
        case LOW:
          return(__timer_ticks_low1);
          break;
        default:
          break;
      }
      break;
      default:
        break;
  }
  return(LOW);
}

// ***********************************************************************************
// void outputPulse1(void)
// 
// Purpose:   Output the different LEVELS of a pulse, including the preceding delay.
//            This routine also takes into account the logic level inversion in place.
// 
// Inputs:    None
// 
// Returns:   Nothing
// ***********************************************************************************
void outputPulse1(void)
{
  if(wait_preambleHigh1) {
    if(time_mark >= (__pulse_mark0 + preambleHigh1)) {
      digitalWrite(outputPin1, OUT_LOW1);       // non-inverted
      __pulse_mark0 = time_mark;
      wait_preambleHigh1 = false;
    }
  }
  else if(wait_delay1) {
    if(time_mark >= (__pulse_mark0 + dwell_delay1)) {
      digitalWrite(outputPin1, OUT_HIGH1);       // non-inverted
      __pulse_mark0 = time_mark;
      wait_delay1 = false;
    }
  }
  else if(wait_high1) {
    if(time_mark >= (__pulse_mark0 + dwell_high1)) {
      digitalWrite(outputPin1, OUT_LOW1);        // non-inverted
      __pulse_mark0 = time_mark;
      wait_high1 = false;
    }
  }
}

// ***********************************************************************************
// void outputPulse2(void)
// 
// Purpose:   Output the different LEVELS of a pulse, including the preceding delay.
//            This routine also takes into account the logic level inversion in place.
// 
// Inputs:    None
// 
// Returns:   Nothing
// ***********************************************************************************
void outputPulse2(void)
{
  if(wait_preambleHigh2) {
    if(time_mark >= (__pulse_mark1 + preambleHigh2)) {
      digitalWrite(outputPin2, OUT_LOW2);       // non-inverted
      __pulse_mark1 = time_mark;
      wait_preambleHigh2 = false;
    }
  }
  else if(wait_delay2) {
    if(time_mark >= (__pulse_mark1 + dwell_delay2)) {
      digitalWrite(outputPin2, OUT_HIGH2);       // non-inverted
      __pulse_mark1 = time_mark;
      wait_delay2 = false;
    }
  }
  else if(wait_high2) {
    if(time_mark >= (__pulse_mark1 + dwell_high2)) {
      digitalWrite(outputPin2, OUT_LOW2);        // non-inverted
      __pulse_mark1 = time_mark;
      wait_high2 = false;
    }
  }
}

// ***********************************************************************************
// void processAnalogInputs(void)
// 
// Purpose:   To sample the voltage appearing on the analog input pins and convert the 
//            read voltage(s) into a percentage in the range: 0V = 0% to 5V = 100%.
//
// Inputs:    None
//
// Returns:   Nothing
// ***********************************************************************************
#if defined(READ_ANALOG_DELAY1) || defined(READ_ANALOG_DELAY1) || defined(READ_ANALOG_DELAY2) || defined(READ_ANALOG_DELAY2)
void processAnalogInputs(void)
{
#ifdef READ_ANALOG_DELAY1
    analog_delay1 = (float)(analogRead(inputAPin1));
    if(analog_delay1 != old_analog_delay1) {
      old_analog_delay1 = analog_delay1;
      dwell_delay_percentage1 = analog_delay1 / 1000.0;
      uldwell_delay_percentage1 = dwell_delay_percentage1 * FMULT;
    }
#endif
#ifdef READ_ANALOG_DELAY2
    analog_delay2 = (float)(analogRead(inputAPin2));
    if(analog_delay2 != old_analog_delay2) {
      old_analog_delay1 = analog_delay1;
      dwell_delay_percentage2 = analog_delay2 / 1000.0;
      uldwell_delay_percentage2 = dwell_delay_percentage2 * FMULT;
    }
#endif

#ifdef READ_ANALOG_DWELL1
    analog_dwell1 = (float)(analogRead(inputAPin3));
    if(analog_dwell1 != old_analog_dwell1) {
      old_analog_dwell1 = analog_dwell1;
      dwell_percentage1 = analog_dwell1 / 1000.0;
      uldwell_percentage1 = dwell_percentage1 * FMULT;
    }
#endif
#ifdef READ_ANALOG_DWELL2
    analog_dwell2 = (float)(analogRead(inputAPin4));
    if(analog_dwell2 != old_analog_dwell2) {
      old_analog_dwell2 = analog_dwell2;
      dwell_percentage2 = analog_dwell2 / 1000.0;
      uldwell_percentage2 = dwell_percentage2 * FMULT;
    }
#endif
}
#endif

// ***********************************************************************************
// void errorCheck(void)
// 
// Purpose:   Perform error checking for missed pulses. If the number of missed pulses
//            or 'hung' pulses is greater than a predetermined limit - do something
//            about it because such a thing as increasing entropy will completely skew
//            the results FAST!
//
// Inputs:    None
//
// Returns:   Nothing
// ***********************************************************************************
void errorCheck(void)
{
  static int error_count1 = 0;
  static int error_count2 = 0;
  
  // *************************************************************
  // Stop interrupts
  // *************************************************************
  cli();

  // *************************************************************
  // Error check: missed pulse on input pin 1
  // *************************************************************
  if(__got_pulse0 && digitalRead(inputPin1) == HIGH) {
    if((time_mark - __pulse_start_time0) > (__iperiod0 << 4)) {
      error_count1++;
      if(error_count1 > MAX_MISSED_PULSES) {
        error_count1 = 0;
        wait_delay1 = false;
        wait_high1 = false;
        Pout1 = false;
        digitalWrite(outputPin1, LOW);
#ifdef RESET_ON_MISSED_PULSES
        sei();
        softReset1();
#endif
      }
    }
  }
  
  // *************************************************************
  // Error check: missed pulse on input pin 2
  // *************************************************************
  if(__got_pulse1 && digitalRead(inputPin2) == HIGH) {
    if((time_mark - __pulse_start_time1) > (__iperiod1 << 4)) {
      error_count2++;
      if(error_count2 > MAX_MISSED_PULSES) {
        error_count2 = 0;
        wait_delay2 = false;
        wait_high2 = false;
        Pout2 = false;
        digitalWrite(outputPin2, LOW);
#ifdef RESET_ON_MISSED_PULSES
        sei();
        softReset1();
#endif
      }
    }
  }

  // *************************************************************
  // Re-start interrupts
  // *************************************************************
  sei();
}

// ***********************************************************************************
// void debug(void)
// 
// Purpose:   Print debug information to the serial port
//
// Inputs:    None
//
// Returns:   Nothing
// ***********************************************************************************
#ifdef DEBUG
void debug(void)
{
#if defined(DEBUG)
  unsigned long dwell_High1 = 0;
  unsigned long dwell_High2 = 0;
  unsigned long dwell_Low1 = 0;
  unsigned long dwell_Low2 = 0;
  float pulseFrequency1 = 0.0;
  float pulseFrequency2 = 0.0;
#endif
  
  // *************************************************************
  // 6. Obtain duration in microseconds of pulse HIGH, LOW levels.
  // *************************************************************
  dwell_High1 = getPulseIn(0, HIGH);
  dwell_Low1 = getPulseIn(0, LOW);
  dwell_High2 = getPulseIn(1, HIGH);
  dwell_Low2 = getPulseIn(1, LOW);
  
  // *************************************************************
  // 7. Calculate pulse period (HIGH + LOW). 
  //    F = 1/T, but T is in microseconds so...
  //    F = (1/T) * 1000000
  // *************************************************************
  pulsePeriod1 = dwell_High1 + dwell_Low1; //pulseWidthHigh1 + pulseWidthLow1;
  pulsePeriod2 = dwell_High2 + dwell_Low2; //pulseWidthHigh2 + pulseWidthLow2;
  pulseFrequency1 = 1000000 / pulsePeriod1;
  pulseFrequency2 = 1000000 / pulsePeriod2;
#endif
  // *************************************************************
  // 8. Output debug information. Note: enabled debug WILL affect the
  //    overall efficiency of the program and should only be used for
  //    reference purposes.
  // *************************************************************
#if defined(DEBUG) || defined(DEBUG_ANALOG_ONLY)
  static long scounter = 5001;
  
  if(scounter >5000) {
    scounter = 0;
#if !defined(DEBUG_ANALOG_ONLY)
    // Output some debugging information...
    IOSERIAL.print(pulseFrequency1, 0);
    IOSERIAL.print(" Hz (out)");  
    IOSERIAL.print(", period = ");
    IOSERIAL.print(pulsePeriod1, DEC);
    IOSERIAL.print(", High = ");
    //IOSERIAL.print(pulseWidthHigh1, DEC);
    IOSERIAL.print(dwell_High1, DEC);
    IOSERIAL.print(", Low = ");
    //IOSERIAL.print(pulseWidthLow1, DEC);
    IOSERIAL.print(dwell_Low1, DEC);
    IOSERIAL.print(", dwell = ");
#ifdef VARIABLE_HIGH
    IOSERIAL.print(dwell_High1, DEC);
#else
    IOSERIAL.print(dwell_High1 << 1, DEC);
#endif
#endif
#if defined(DEBUG) && !defined(DEBUG_ANALOG_ONLY)
#if defined(READ_ANALOG_DWELL1) || defined(READ_ANALOG_DELAY1) || defined(READ_ANALOG_DWELL2) || defined(READ_ANALOG_DELAY2)
    IOSERIAL.print(",");
#endif
#endif
#if defined(DEBUG_ANALOG_ONLY) || (defined(DEBUG) && !defined(DEBUG_ANALOG_ONLY))
#ifdef READ_ANALOG_DWELL1
    IOSERIAL.print(" dwell1 = ");
    IOSERIAL.print(dwell_percentage1 * 100, 0);
    IOSERIAL.print("%");
#endif
#if defined(READ_ANALOG_DELAY1) || defined(READ_ANALOG_DWELL2) || defined(READ_ANALOG_DELAY2)
    IOSERIAL.print(",");
#endif
#ifdef READ_ANALOG_DELAY1
    IOSERIAL.print(" delay1 = ");
    IOSERIAL.print(dwell_delay_percentage1 * 100, 0);
    IOSERIAL.print("%");
#endif
#if defined(READ_ANALOG_DWELL2) || defined(READ_ANALOG_DELAY2)
    IOSERIAL.print(",");
#endif
#ifdef READ_ANALOG_DWELL2
    IOSERIAL.print(" dwell2 = ");
    IOSERIAL.print(dwell_percentage2 * 100, 0);
    IOSERIAL.print("%");
#endif
#if defined(READ_ANALOG_DELAY2)
    IOSERIAL.print(",");
#endif
#ifdef READ_ANALOG_DELAY2
    IOSERIAL.print(" delay2 = ");
    IOSERIAL.print(dwell_delay_percentage2 * 100, 0);
    IOSERIAL.print("%");
#endif
#endif
#ifdef VARIABLE_HIGH
    IOSERIAL.print(" (variable HIGH)");
#endif
    IOSERIAL.println(" ");
  }
  scounter++;
}
#endif

// ***********************************************************************************
// void loop(void)
// 
// Purpose:   Main program loop. The work flow in this loop is as follows:
//
//            1. Read analog values and convert to percentages
//            2. Fetch the current time for use in the next section
//            3. See if any pulses are pending and if so, output the correct
//               levels depending how much time has elapsed
//            4. Error check: missed pulse on input pin 1
//            5. Handle user input from the keyboard
//            6. Obtain duration in microseconds of pulse HIGH, LOW levels.
//            7. Calculate pulse period (HIGH + LOW). 
//               F = 1/T, but T is in microseconds so...
//               F = (1/T) * 1000000
//            8. Output debug information. Note: enabled debug WILL affect the
//               overall efficiency of the program and should only be used for
//               reference purposes.
//
// Inputs:    None
//
// Returns:   Nothing
// ***********************************************************************************
void loop(void)
{
  // *************************************************************
  // 1. Read analog values and convert to percentages
  // *************************************************************
#if defined(READ_ANALOG_DELAY1) || defined(READ_ANALOG_DELAY1) || defined(READ_ANALOG_DELAY2) || defined(READ_ANALOG_DELAY2)
  processAnalogInputs();
#endif

  // *************************************************************
  // 2. Fetch the current time for use in the next section
  // *************************************************************
  time_mark = micros();

  // *************************************************************
  // 3. See if any pulses are pending and if so, output the correct
  //    levels depending how much time has elapsed
  // *************************************************************
  outputPulse1();
  outputPulse2();
  
  // *************************************************************
  // 4. Error check: missed pulse on input pin 1
  // *************************************************************
  //errorCheck();
  
  // *************************************************************
  // 5. Handle user input from the keyboard
  // *************************************************************
#ifdef  USE_KEYBOARD
  if(keyboard.available()) {
    char c = keyboard.read();
    switch(c) {
      case 'q':
        if((int)(dwell_delay_percentage1 * 100.0) < 99) {
          dwell_delay_percentage1 = dwell_delay_percentage1 + 0.01;
          uldwell_delay_percentage1 = (unsigned long)(dwell_delay_percentage1 * FMULT);
#ifdef USE_KEYBOARD_STATS
          IOSERIAL.print("Delay [1] ");
          IOSERIAL.print((dwell_delay_percentage1 * 100), 2);
          IOSERIAL.println("%");
#endif
        }
        break;
      case 'Q':
        if((int)(dwell_percentage1 * 100.0) < 99) {
          dwell_percentage1 = dwell_percentage1 + 0.01;
          uldwell_percentage1 = (unsigned long)(dwell_percentage1 * FMULT);
#ifdef USE_KEYBOARD_STATS
          IOSERIAL.print("Dwell [1] ");
          IOSERIAL.print((dwell_percentage1 * 100), 2);
          IOSERIAL.println("%");
#endif
        }
        break;
      case 'a':
        if((int)(dwell_delay_percentage1 * 100.0) > 1) {
          dwell_delay_percentage1 = dwell_delay_percentage1 - 0.01; 
          uldwell_delay_percentage1 = (unsigned long)(dwell_delay_percentage1 * FMULT);
#ifdef USE_KEYBOARD_STATS
          IOSERIAL.print("Delay [1] ");
          IOSERIAL.print((dwell_delay_percentage1 * 100), 2);
          IOSERIAL.println("%");
#endif
        }
        break;
      case 'A':
        if((int)(dwell_percentage1 * 100.0) > 1) {
          dwell_percentage1 = dwell_percentage1 - 0.01; 
          uldwell_percentage1 = (unsigned long)(dwell_percentage1 * FMULT);
#ifdef USE_KEYBOARD_STATS
          IOSERIAL.print("Dwell [1] ");
          IOSERIAL.print((dwell_percentage1 * 100), 2);
          IOSERIAL.println("%");
#endif
        }
        break;
      case 'w':
        if((int)(dwell_delay_percentage2 * 100.0) < 99) {
          dwell_delay_percentage2 = dwell_delay_percentage2 + 0.01;
          uldwell_delay_percentage2 = (unsigned long)(dwell_delay_percentage2 * FMULT);
#ifdef USE_KEYBOARD_STATS
          IOSERIAL.print("Delay [2] ");
          IOSERIAL.print((dwell_delay_percentage2 * 100), 2);
          IOSERIAL.println("%");
#endif
        }
        break;
      case 'W':
        if((int)(dwell_percentage2 * 100.0) < 99) {
          dwell_percentage2 = dwell_percentage2 + 0.01;
          uldwell_percentage2 = (unsigned long)(dwell_percentage2 * FMULT);
#ifdef USE_KEYBOARD_STATS
          IOSERIAL.print("Dwell [2] ");
          IOSERIAL.print((dwell_percentage2 * 100), 2);
          IOSERIAL.println("%");
#endif
        }
        break;
      case 's':
        if((int)(dwell_delay_percentage2 * 100.0) > 1) {
          dwell_delay_percentage2 = dwell_delay_percentage2 - 0.01; 
          uldwell_delay_percentage2 = (unsigned long)(dwell_delay_percentage2 * FMULT);
#ifdef USE_KEYBOARD_STATS
          IOSERIAL.print("Delay [2] ");
          IOSERIAL.print((dwell_delay_percentage2 * 100), 2);
          IOSERIAL.println("%");
#endif
        }
        break;
      case 'S':
        if((int)(dwell_percentage2 * 100.0) > 1) {
          dwell_percentage2 = dwell_percentage2 - 0.01; 
          uldwell_percentage2 = (unsigned long)(dwell_percentage2 * FMULT);
#ifdef USE_KEYBOARD_STATS
          IOSERIAL.print("Dwell [2] ");
          IOSERIAL.print((dwell_percentage2 * 100), 2);
          IOSERIAL.println("%");
#endif
        }
        break;
      case 'p':
      case 'P':
#ifdef USE_KEYBOARD_STATS
        IOSERIAL.println(" ");
        IOSERIAL.println("Muller Motor Driver Settings Report");
        IOSERIAL.print("Dwell [1] ");
        IOSERIAL.print((dwell_percentage1 * 100), 2);
        IOSERIAL.println("%");
        IOSERIAL.print("Delay [1] ");
        IOSERIAL.print((dwell_delay_percentage1 * 100), 2);
        IOSERIAL.println("%");
        IOSERIAL.print("Dwell [2] ");
        IOSERIAL.print((dwell_percentage2 * 100), 2);
        IOSERIAL.println("%");
        IOSERIAL.print("Delay [2] ");
        IOSERIAL.print((dwell_delay_percentage2 * 100), 2);
        IOSERIAL.println("%");
#endif
        break;
      case 'r':
      case 'R':
#ifdef USE_KEYBOARD_STATS
        IOSERIAL.println("Reset:");
#endif
        softReset1();
        break;
      default:
        break;
    }
  }
#endif

#ifdef DEBUG
  debug();
#endif
}

// ***********************************************************************************
// int main(void)
// 
// Purpose:   Traditional 'C' programs require a user supplied function called 'main'.
//            This function is conditionaly supplied here because the AVR Studio
//            environment follows the model of traditional 'C' programming while the
//            Arduino IDE - targeting beginner and non-professional developers - omits
//            this compiler requirement (but nevertheless provides it under the hood 
//            so to speak). 'main()' is the actual entry point for the user program.
//
// Inputs:    None
//
// Returns:   0 (error code)
// ***********************************************************************************
#ifdef AVR_STUDIO
int main(void)
{
  init();
  
  setup();
  
  while(1) {
    loop();
  }
  
  return 0;
}
#endif
